from typing import List, Optional, Tuple

from axelrod._strategy_utils import detect_cycle

from axelrod.action import Action

from axelrod.player import Player

C, D = Action.C, Action.D

def is_alternator(history: List[Action]) -> bool:
    for i in range(len(history) - 1):
        if history[i] == history[i + 1]:
            return False
    return True

class CycleHunter(Player):
    """Hunts strategies that play cyclically, like any of the Cyclers,
    Alternator, etc.

    Names:

    - Cycle Hunter: Original name by Marc Harper
    """

    name = "Cycle Hunter"
    classifier = {
        "memory_depth": float("inf"),  # Long memory
        "stochastic": False,
        "long_run_time": False,
        "inspects_source": False,
        "manipulates_source": False,
        "manipulates_state": False,
    }

    def __init__(self) -> None:
        super().__init__()
        self.cycle = None  # type: Optional[Tuple[Action]]

    def strategy(self, opponent: Player) -> Action:
        """Actual strategy definition that determines player's action."""
        if self.cycle:
            return D
        cycle = detect_cycle(opponent.history, min_size=3)
        if cycle:
            if len(set(cycle)) > 1:
                self.cycle = cycle
                return D
        return C